package com.gravatar.quickeditor.ui.abouteditor

import app.cash.turbine.test
import com.gravatar.extensions.defaultProfile
import com.gravatar.quickeditor.data.models.QuickEditorError
import com.gravatar.quickeditor.data.repository.ProfileRepository
import com.gravatar.quickeditor.ui.CoroutineTestRule
import com.gravatar.quickeditor.ui.avatarpicker.SectionError
import com.gravatar.quickeditor.ui.editor.AboutInputField
import com.gravatar.restapi.models.UpdateProfileRequest
import com.gravatar.services.ErrorType
import com.gravatar.services.GravatarResult
import com.gravatar.types.Email
import io.mockk.coEvery
import io.mockk.mockk
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.test.StandardTestDispatcher
import kotlinx.coroutines.test.advanceUntilIdle
import kotlinx.coroutines.test.runTest
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Rule
import org.junit.Test

@OptIn(ExperimentalCoroutinesApi::class)
class AboutEditorViewModelTest {
    private val testDispatcher = StandardTestDispatcher()

    @get:Rule
    var coroutineTestRule = CoroutineTestRule(testDispatcher)

    private val profileRepository = mockk<ProfileRepository>()
    private lateinit var viewModel: AboutEditorViewModel

    private val email = Email("testEmail")
    private val profile = defaultProfile(hash = "hash")
    private val updatedProfile = defaultProfile(hash = "hash", displayName = "Updated Name")
    private val updateProfileRequest = UpdateProfileRequest {
        firstName = ""
        lastName = ""
        displayName = "Updated Name"
        jobTitle = ""
        company = ""
        description = ""
        location = ""
        pronunciation = ""
        pronouns = ""
    }
    private val visibleAboutFields = AboutInputField.all

    @Before
    fun setup() {
        coEvery { profileRepository.getProfile(email) } returns GravatarResult.Success(profile)
    }

    @Test
    fun `given view model initialization when profile fetch succeeds then uiState is updated`() = runTest {
        viewModel = initViewModel()

        viewModel.uiState.test {
            assertEquals(AboutEditorUiState(isLoading = false), awaitItem())
            assertEquals(AboutEditorUiState(isLoading = true), awaitItem())
            val aboutFields = profile.aboutFields(visibleAboutFields)
            assertEquals(
                AboutEditorUiState(
                    isLoading = false,
                    aboutFields = aboutFields,
                    savedAboutFields = aboutFields,
                ),
                awaitItem(),
            )
        }
    }

    @Test
    fun `given view model initialization when profile fetch fails then uiState is updated`() = runTest {
        coEvery { profileRepository.getProfile(email) } returns GravatarResult.Failure(
            QuickEditorError.Unknown,
        )

        viewModel = initViewModel()

        viewModel.uiState.test {
            assertEquals(AboutEditorUiState(isLoading = false), awaitItem())
            assertEquals(AboutEditorUiState(isLoading = true), awaitItem())
            assertEquals(
                AboutEditorUiState(
                    isLoading = false,
                    error = SectionError.Unknown,
                ),
                awaitItem(),
            )
        }
    }

    @Test
    fun `given view model initialization when profile fetch fails with unauthorized then uiState is updated`() =
        runTest {
            coEvery { profileRepository.getProfile(email) } returns GravatarResult.Failure(
                QuickEditorError.Request(ErrorType.Unauthorized),
            )

            viewModel = initViewModel()

            viewModel.uiState.test {
                assertEquals(AboutEditorUiState(isLoading = false), awaitItem())
                assertEquals(AboutEditorUiState(isLoading = true), awaitItem())
                assertEquals(
                    AboutEditorUiState(
                        isLoading = false,
                        error = SectionError.InvalidToken(true),
                    ),
                    awaitItem(),
                )
            }
        }

    @Test
    fun `given updated about field when save clicked and update succeeds then uiState is updated`() = runTest {
        coEvery {
            profileRepository.updateProfile(email, updateProfileRequest)
        } returns GravatarResult.Success(updatedProfile)

        viewModel = initViewModel()
        viewModel.onEvent(
            AboutEditorEvent.OnAboutFieldUpdated(
                AboutEditorField(
                    type = AboutInputField.DisplayName,
                    value = "Updated Name",
                ),
            ),
        )

        advanceUntilIdle()

        viewModel.onEvent(AboutEditorEvent.OnSaveClicked)

        viewModel.uiState.test {
            expectMostRecentItem()
            val aboutFields = updatedProfile.aboutFields(visibleAboutFields)
            assertEquals(
                AboutEditorUiState(
                    savingProfile = true,
                    aboutFields = aboutFields,
                    savedAboutFields = profile.aboutFields(visibleAboutFields),
                ),
                awaitItem(),
            )
            assertEquals(
                AboutEditorUiState(
                    savingProfile = false,
                    aboutFields = aboutFields,
                    savedAboutFields = aboutFields,
                ),
                awaitItem(),
            )
        }
        viewModel.actions.test {
            assertEquals(AboutEditorAction.ProfileUpdated(updatedProfile), awaitItem())
        }
    }

    @Test
    fun `given updated about field when save clicked and update fails then uiState is updated`() = runTest {
        coEvery {
            profileRepository.updateProfile(email, updateProfileRequest)
        } returns GravatarResult.Failure(QuickEditorError.Unknown)

        viewModel = initViewModel()
        viewModel.onEvent(
            AboutEditorEvent.OnAboutFieldUpdated(
                AboutEditorField(
                    type = AboutInputField.DisplayName,
                    value = "Updated Name",
                ),
            ),
        )

        advanceUntilIdle()

        viewModel.onEvent(AboutEditorEvent.OnSaveClicked)

        viewModel.uiState.test {
            expectMostRecentItem()
            val aboutFields = updatedProfile.aboutFields(visibleAboutFields)
            assertEquals(
                AboutEditorUiState(
                    savingProfile = true,
                    aboutFields = aboutFields,
                    savedAboutFields = profile.aboutFields(visibleAboutFields),
                ),
                awaitItem(),
            )
            assertEquals(
                AboutEditorUiState(
                    savingProfile = false,
                    aboutFields = aboutFields,
                    savedAboutFields = profile.aboutFields(visibleAboutFields),
                ),
                awaitItem(),
            )
        }
        viewModel.actions.test {
            assertEquals(AboutEditorAction.ProfileUpdateFailed, awaitItem())
        }
    }

    @Test
    fun `given updated about field when save clicked and update fails with 401 then uiState is updated`() = runTest {
        coEvery {
            profileRepository.updateProfile(email, updateProfileRequest)
        } returns GravatarResult.Failure(QuickEditorError.Request(ErrorType.Unauthorized))

        viewModel = initViewModel()
        viewModel.onEvent(
            AboutEditorEvent.OnAboutFieldUpdated(
                AboutEditorField(
                    type = AboutInputField.DisplayName,
                    value = "Updated Name",
                ),
            ),
        )

        advanceUntilIdle()

        viewModel.onEvent(AboutEditorEvent.OnSaveClicked)

        viewModel.uiState.test {
            val aboutFields = updatedProfile.aboutFields(visibleAboutFields)
            expectMostRecentItem()
            val savedAboutFields = profile.aboutFields(visibleAboutFields)
            assertEquals(
                AboutEditorUiState(
                    savingProfile = true,
                    aboutFields = aboutFields,
                    savedAboutFields = savedAboutFields,
                ),
                awaitItem(),
            )
            assertEquals(
                AboutEditorUiState(
                    savingProfile = false,
                    aboutFields = aboutFields,
                    savedAboutFields = savedAboutFields,
                    error = SectionError.InvalidToken(
                        showLogin = true,
                    ),
                ),
                awaitItem(),
            )
        }
        viewModel.actions.test {
            expectNoEvents()
        }
    }

    @Test
    fun `given only personal about section visible when profile fetched then uiState is updated`() = runTest {
        viewModel = initViewModel(aboutFields = AboutInputField.personal)

        viewModel.uiState.test {
            assertEquals(AboutEditorUiState(isLoading = false), awaitItem())
            assertEquals(AboutEditorUiState(isLoading = true), awaitItem())
            assertEquals(
                setOf(
                    AboutEditorField(
                        type = AboutInputField.DisplayName,
                        value = profile.displayName,
                        maxLines = 1,
                    ),
                    AboutEditorField(
                        type = AboutInputField.AboutMe,
                        value = profile.description,
                        maxLines = 4,
                    ),
                    AboutEditorField(
                        type = AboutInputField.Pronunciation,
                        value = profile.pronunciation,
                    ),
                    AboutEditorField(
                        type = AboutInputField.Pronouns,
                        value = profile.pronouns,
                    ),
                    AboutEditorField(
                        type = AboutInputField.Location,
                        value = profile.location,
                    ),
                ),
                awaitItem().aboutFields,
            )
        }
    }

    @Test
    fun `given no changes when state update then save button is disabled`() = runTest {
        viewModel = initViewModel()

        advanceUntilIdle()

        viewModel.uiState.test {
            assertEquals(false, awaitItem().saveEnabled)
        }
    }

    @Test
    fun `given changes when state update then save button is enabled`() = runTest {
        viewModel = initViewModel()

        viewModel.onEvent(
            AboutEditorEvent.OnAboutFieldUpdated(
                AboutEditorField(
                    type = AboutInputField.DisplayName,
                    value = "New Display Name",
                ),
            ),
        )

        advanceUntilIdle()

        viewModel.uiState.test {
            assertEquals(true, awaitItem().saveEnabled)
        }
    }

    private fun initViewModel(aboutFields: Set<AboutInputField> = visibleAboutFields) = AboutEditorViewModel(
        email = email,
        profileRepository = profileRepository,
        visibleAboutFields = aboutFields,
        handleExpiredSession = true,
    )
}
