package com.gravatar.quickeditor.ui.components

import android.content.res.Configuration
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.ripple.rememberRipple
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.minimumInteractiveComponentSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.dp
import com.gravatar.AvatarQueryOptions
import com.gravatar.DefaultAvatarOption
import com.gravatar.ImageRating
import com.gravatar.extensions.defaultProfile
import com.gravatar.quickeditor.R
import com.gravatar.restapi.models.Profile
import com.gravatar.types.Email
import com.gravatar.ui.GravatarTheme
import com.gravatar.ui.components.ComponentState
import com.gravatar.ui.components.ProfileSummary
import com.gravatar.ui.components.atomic.Avatar
import com.gravatar.ui.components.atomic.ViewProfileButton

@Composable
internal fun ProfileCard(
    profile: ComponentState<Profile>?,
    email: Email,
    modifier: Modifier = Modifier,
    editAvatarEnabled: Boolean = false,
    editAboutEnabled: Boolean = false,
    onEditAvatarClicked: () -> Unit = { },
    onEditAboutClicked: () -> Unit = { },
    avatarCacheBuster: String? = null,
) {
    CompositionLocalProvider(
        LocalDensity provides Density(
            LocalDensity.current.density,
            LocalDensity.current.fontScale.coerceAtMost(1.5f),
        ),
    ) {
        GravatarCard(modifier) { backgroundColor ->
            profile?.let {
                Row(
                    verticalAlignment = Alignment.Top,
                    modifier = Modifier
                        .fillMaxWidth()
                        .background(backgroundColor)
                        .padding(horizontal = 16.dp, vertical = 11.dp),
                ) {
                    Box(
                        modifier = Modifier
                            .weight(1f, fill = true),
                    ) {
                        ProfileSummary(
                            state = it,
                            modifier = Modifier
                                .background(backgroundColor),
                            avatar = {
                                val sizePx = with(LocalDensity.current) { 72.dp.roundToPx() }
                                Box {
                                    Avatar(
                                        email = email,
                                        avatarQueryOptions = AvatarQueryOptions {
                                            preferredSize = sizePx
                                            rating = ImageRating.X
                                            defaultAvatarOption = DefaultAvatarOption.Status404
                                        },
                                        size = 72.dp,
                                        modifier = Modifier.clip(CircleShape),
                                        cacheBuster = avatarCacheBuster,
                                    )
                                    if (editAvatarEnabled) {
                                        EditButton(
                                            onClick = onEditAvatarClicked,
                                            backgroundColor = backgroundColor,
                                            contentAlignment = Alignment.BottomEnd,
                                            contentDescription = stringResource(
                                                R.string.gravatar_qe_edit_avatar_content_description,
                                            ),
                                            modifier = Modifier
                                                .offset(12.dp, 12.dp)
                                                .align(Alignment.BottomEnd),
                                        )
                                    }
                                }
                            },
                            viewProfile = { state ->
                                if (state !is ComponentState.Empty) {
                                    ViewProfileButton(
                                        state = state,
                                        modifier = Modifier.height(32.dp),
                                    ) {
                                        Icon(
                                            painterResource(R.drawable.ic_external),
                                            tint = MaterialTheme.colorScheme.onBackground,
                                            contentDescription = "",
                                        )
                                    }
                                }
                            },
                        )
                    }
                    if (editAboutEnabled) {
                        EditButton(
                            onClick = onEditAboutClicked,
                            backgroundColor = backgroundColor,
                            contentAlignment = Alignment.TopEnd,
                            contentDescription = stringResource(
                                R.string.gravatar_qe_edit_about_content_description,
                            ),
                            modifier = Modifier
                                .offset(x = 12.dp, y = (-12).dp)
                                .padding(top = 5.dp),
                        )
                    }
                }
            }
        }
    }
}

@Composable
internal fun GravatarCard(modifier: Modifier = Modifier, content: @Composable (Color) -> Unit) {
    val cardModifier = modifier
        .wrapContentHeight()
    val shape = RoundedCornerShape(8.dp)
    if (isSystemInDarkTheme()) {
        Card(
            modifier = cardModifier,
            shape = shape,
        ) {
            content(MaterialTheme.colorScheme.surfaceContainerHigh)
        }
    } else {
        ElevatedCard(
            modifier = cardModifier,
            elevation = CardDefaults.elevatedCardElevation(defaultElevation = 6.dp),
            shape = shape,
        ) {
            content(MaterialTheme.colorScheme.surface)
        }
    }
}

@Composable
private fun EditButton(
    onClick: () -> Unit,
    backgroundColor: Color,
    contentDescription: String?,
    modifier: Modifier = Modifier,
    contentAlignment: Alignment = Alignment.Center,
) {
    Box(
        modifier = modifier
            .minimumInteractiveComponentSize()
            .clickable(
                onClick = onClick,
                interactionSource = remember { MutableInteractionSource() },
                indication = rememberRipple(
                    bounded = false,
                    radius = 48.dp / 2,
                ),
            ),
        contentAlignment = contentAlignment,
    ) {
        Icon(
            painter = painterResource(R.drawable.ic_edit),
            contentDescription = contentDescription,
            tint = MaterialTheme.colorScheme.onSurface,
            modifier = Modifier
                .size(24.dp)
                .clip(CircleShape)
                .background(backgroundColor)
                .padding(2.dp),
        )
    }
}

@Preview(showBackground = true, heightDp = 200)
@Preview(uiMode = Configuration.UI_MODE_NIGHT_YES, showBackground = true, backgroundColor = 0xFF000000, heightDp = 200)
@Composable
private fun ProfileCardPreview() {
    GravatarTheme {
        ProfileCard(
            profile = ComponentState.Loaded(
                defaultProfile(hash = "dfadf", "John Travolta"),
            ),
            email = Email("john.adams@test.com"),
            modifier = Modifier.padding(20.dp),
            editAvatarEnabled = true,
            editAboutEnabled = true,
        )
    }
}
